function [poslist] = join_poslist(varargin)
%This function allows you to join imaging position lists for axiovision in
%either interleave (Default) or end to end mode
%Synatax:   [poslist] = join_poslist('dir1','c:\...','files','syn.csv','mode',1)
%Input:     'path' = the directory where the files are
%           'files' = the position filenames in a cell array.
%           'mode' = either interleave mode (Default = 1) or end to end
%               mode (0).  Note: Interleave mode only works with files that
%               have the same number of positions.  Now for mosaic position
%               lists join interleaved using mode = 3; Mode 3 not fully
%               functional currently.
%Output:    poslist = the joined position list.

[pathname,files,md] = parse(varargin);  %parse the input

outfile = files{1}(1:end-4);   %initiate
for i = 1:size(files,2)
    fid = fopen([pathname,filesep,files{i}]);  %open the file
    data_raw = textscan(fid,'%s %s %s %s %s %s','Delimiter',',');   %grab the raw text data
    data_tmp = [];  %initiate
    for j = 1:size(data_raw,2)      %combine the raw data into a single file
%         if i>1 && j==1  %the first column of files not the first
%             cache = char(data_raw{1,j});    %create a character array so we can manipulate it
%             cache(:,2) = num2str(i);    %change the numbering
%             data_raw{1,j} = cellstr(cache);     %convert back
%         end
        data_tmp = horzcat(data_tmp,data_raw{1,j});
    end
    %first find the position where the data 
    ptr = 1;    %pointer
    while size(data_tmp{ptr,1},2)<=2||isnan(str2double(data_tmp{ptr,1}(3)))     %Keep going until we hit a number
        ptr = ptr+1;
    end
    %Now strip the header, store it if it is the first file
    if i==1
        header = data_tmp(1:ptr-1,:);
    end
    data_tmp = data_tmp(ptr:end,:);
    %store the data
    lists(i).data = data_tmp;
    %also create an output filename
    if i>1
        outfile = [outfile,'+',files{i}(1:end-4)];
    end
end
%now renumber the fields
[lists] = parsepositions(lists,size(files,2),md);

%now combine the files
switch md
    case 0  %end to end
        poslist = lists(1).data;    %grab the first file
        for l = 2:size(files,2)     %go through and append each file
            tmp = lists(l).data;
            if rem(l,2)     %if it is odd don't flip
                poslist = vertcat(poslist,tmp(1:end,:));
            else    %even files flip
                poslist = vertcat(poslist,flipud(tmp(1:end,:)));
            end
        end
        outfile = [outfile,'_','endjoined'];  %specifiy the type of join
    case 3  %for multifield datasets, interleave
        %we must be section aware, use the section index
        poslist = [];
        for k = 1:max(lists(1).sectidx)  %go through each section
            poscache = [];
            for l = 1:size(files,2)     %pull section data from each file set
                posidx = lists(l).sectidx==k;  %all position indexs of a section
                postmp = lists(l).data(posidx,:);     %all positions of a section
                poscache = vertcat(poscache,postmp);     %put data together
            end
            poslist = vertcat(poslist,poscache);
        end        
        outfile = [outfile,'_','interleave'];  %specifiy the type of join
    otherwise   %interleaved
        poslist = [];
        for k = 1:size(data_tmp,1)
            for l = 1:size(files,2)     %go through the files
                poslist = vertcat(poslist,lists(l).data(k,:));  %put in each position
            end
        end
        outfile = [outfile,'_','interleave'];  %specifiy the type of join
end
%put on the header
poslist = vertcat(header,poslist);
%save out the file
sav2csv(poslist,outfile,pathname);

%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [pathname,files,md] = parse(input)

pathname = [];
files = [];
md = 1;

%Parse the input
if ~isempty(input)
    for i = 1:2:size(input,2)
        if ischar(input{1,i});
            switch input{1,i}
                case 'path'
                    pathname = input{1,i+1};
                case 'files'
                    files = input{1,i+1};
                case 'mode'
                    md = input{1,i+1};
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        else
            error(['The parameters you entered is incorrect.  Please check help.']);
        end
    end
end

if isempty(pathname)||isempty(files)   %get files and directory
    [files,pathname,filterindex] = uigetfile2({'*.csv','Text files (*.csv)';'*.*','All Files';},...
        'Open position lists','Multiselect','on');
    if ~iscell(files)   %make sure the files array is a cell array
        files = {files};
    end
end

%--------------------------------------------------------------------------
function [lists] = parsepositions(lists,num,md)
%this takes in the data and renumbers the positions by their section,
%region and field.  num = number of files.  md = the mode
for i = 1:num   %iterate the datasets
    regnum = i;     %the region number is the file number
    switch md
        case 3    %if multifields, act diffent
            for j = 1:size(lists(i).data,1)
                %first we must parse the field string
                datastr = lists(i).data{j,1}(2:end-1);
                %find S & F first for the section number and field numbers
                sloc = strfind(datastr,'S');
                floc = strfind(datastr,'F');
                %now grab the field and section numbers
                sectnum = str2double(datastr(sloc+1:floc-2));   %assuming the datafield is like "S000_F000"
                sectidx(j,1) = sectnum;     %store the section number for later
                fieldnum = str2double(datastr(floc+1:end));
                [section,region,field] = padnum(sectnum,regnum,fieldnum);
                numstrtmp = ['"S',section,'_','R',region,'_','F',field,'"'];
                %now put back
                lists(i).data{j,1} = numstrtmp;
            end
            lists(i).sectidx = sectidx;     %store the section index;
            clear sectidx;
        otherwise   %otherwise default for single field
            for j = 1:size(lists(i).data,1)     %go through each field
                fieldnum = str2double(lists(i).data{j,1}(3:end-1));    %grab the field number
                sectnum = j;    %the section, for single fields, is the field number
                %now create a new string for the data number
                %first pad the numbers if needed
                [section,region,field] = padnum(sectnum,regnum,fieldnum);
                %OK now what we are going to do is to forget the actual
                %field numbers, and use the region for field, because since
                %this is a single field each region is essentially a field.
                %numstrtmp = ['"S',section,'_','R',region,'_','F',field,'"'];
                numstrtmp = ['"S',section,'_','F',region,'"'];
                %now put back
                lists(i).data{j,1} = numstrtmp;
            end
    end
end
%--------------------------------------------------------------------------
function [section,region,field] = padnum(sectnum,regnum,fieldnum)
%pad the numbers 
%section
if sectnum<10   %double pad
    section = ['00',num2str(sectnum)];   
elseif sectnum<100  %single pad
    section = ['0',num2str(sectnum)];
else        %no pad
    section = num2str(sectnum);
end
%region
if regnum<10   %double pad
    region = ['00',num2str(regnum)];   
elseif regnum<100  %single pad
    region = ['0',num2str(regnum)];
else        %no pad
    region = num2str(regnum);
end
%field
if fieldnum<10   %double pad
    field = ['00',num2str(fieldnum)];   
elseif fieldnum<100  %single pad
    field = ['0',num2str(fieldnum)];
else        %no pad
    field = num2str(fieldnum);
end